<?php
/**
 * AJAX Handler Class.
 *
 * Handles all AJAX requests for the plugin.
 *
 * @package WP_WebP_Optimizer
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * AJAX Handler Class.
 */
class WP_WebP_Optimizer_Ajax_Handler {

	/**
	 * Initialize AJAX handlers.
	 *
	 * @since 1.0.0
	 */
	public function init() {
		// Bulk conversion actions
		add_action( 'wp_ajax_webp_start_bulk_conversion', array( $this, 'start_bulk_conversion' ) );
		add_action( 'wp_ajax_webp_process_batch', array( $this, 'process_batch' ) );
		add_action( 'wp_ajax_webp_pause_conversion', array( $this, 'pause_conversion' ) );
		add_action( 'wp_ajax_webp_resume_conversion', array( $this, 'resume_conversion' ) );
		add_action( 'wp_ajax_webp_stop_conversion', array( $this, 'stop_conversion' ) );
		
		// Stats and info
		add_action( 'wp_ajax_webp_get_stats', array( $this, 'get_stats' ) );
		add_action( 'wp_ajax_webp_get_progress', array( $this, 'get_progress' ) );
		
		// Tools
		add_action( 'wp_ajax_webp_test_conversion', array( $this, 'test_conversion' ) );
		add_action( 'wp_ajax_webp_check_server', array( $this, 'check_server' ) );
		add_action( 'wp_ajax_webp_restore_originals', array( $this, 'restore_originals' ) );
		add_action( 'wp_ajax_webp_clear_logs', array( $this, 'clear_logs' ) );
		add_action( 'wp_ajax_webp_export_logs', array( $this, 'export_logs' ) );
		
		// Single image conversion
		add_action( 'wp_ajax_webp_convert_single', array( $this, 'convert_single' ) );
	}

	/**
	 * Start bulk conversion process.
	 *
	 * @since 1.0.0
	 */
	public function start_bulk_conversion() {
		check_ajax_referer( 'webp_bulk_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'Mak8it.com' ) ) );
		}

		require_once SMALL_IMAGE_PLUGIN_DIR . 'includes/class-bulk-processor.php';
		$processor = new WP_WebP_Optimizer_Bulk_Processor();

		// Get filters from request
		$filters = isset( $_POST['filters'] ) ? $_POST['filters'] : array();

		// Get images to convert
		$image_ids = $processor->get_unconverted_images( $filters );

		if ( empty( $image_ids ) ) {
			wp_send_json_error( array( 'message' => __( 'No images to convert', 'Mak8it.com' ) ) );
		}

		// Initialize bulk conversion
		$processor->initialize_bulk_conversion( $image_ids );

		wp_send_json_success( array(
			'message' => __( 'Bulk conversion started', 'Mak8it.com' ),
			'total'   => count( $image_ids ),
		) );
	}

	/**
	 * Process next batch.
	 *
	 * @since 1.0.0
	 */
	public function process_batch() {
		check_ajax_referer( 'webp_bulk_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'Mak8it.com' ) ) );
		}

		require_once SMALL_IMAGE_PLUGIN_DIR . 'includes/class-bulk-processor.php';
		$processor = new WP_WebP_Optimizer_Bulk_Processor();

		$result = $processor->process_next_batch();

		if ( $result['success'] ) {
			wp_send_json_success( $result );
		} else {
			wp_send_json_error( $result );
		}
	}

	/**
	 * Pause conversion.
	 *
	 * @since 1.0.0
	 */
	public function pause_conversion() {
		check_ajax_referer( 'webp_bulk_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'Mak8it.com' ) ) );
		}

		require_once SMALL_IMAGE_PLUGIN_DIR . 'includes/class-bulk-processor.php';
		$processor = new WP_WebP_Optimizer_Bulk_Processor();
		$processor->pause_conversion();

		wp_send_json_success( array( 'message' => __( 'Conversion paused', 'Mak8it.com' ) ) );
	}

	/**
	 * Resume conversion.
	 *
	 * @since 1.0.0
	 */
	public function resume_conversion() {
		check_ajax_referer( 'webp_bulk_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'Mak8it.com' ) ) );
		}

		require_once SMALL_IMAGE_PLUGIN_DIR . 'includes/class-bulk-processor.php';
		$processor = new WP_WebP_Optimizer_Bulk_Processor();
		$processor->resume_conversion();

		wp_send_json_success( array( 'message' => __( 'Conversion resumed', 'Mak8it.com' ) ) );
	}

	/**
	 * Stop conversion.
	 *
	 * @since 1.0.0
	 */
	public function stop_conversion() {
		check_ajax_referer( 'webp_bulk_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'Mak8it.com' ) ) );
		}

		require_once SMALL_IMAGE_PLUGIN_DIR . 'includes/class-bulk-processor.php';
		$processor = new WP_WebP_Optimizer_Bulk_Processor();
		$processor->stop_conversion();

		wp_send_json_success( array( 'message' => __( 'Conversion stopped', 'Mak8it.com' ) ) );
	}

	/**
	 * Get current statistics.
	 *
	 * @since 1.0.0
	 */
	public function get_stats() {
		check_ajax_referer( 'webp_bulk_nonce', 'nonce' );

		require_once SMALL_IMAGE_PLUGIN_DIR . 'includes/class-stats.php';
		$stats = WP_WebP_Optimizer_Stats::get_stats();

		wp_send_json_success( $stats );
	}

	/**
	 * Get conversion progress.
	 *
	 * @since 1.0.0
	 */
	public function get_progress() {
		check_ajax_referer( 'webp_bulk_nonce', 'nonce' );

		$progress = get_option( 'wp_webp_conversion_progress', array() );
		$status = get_option( 'wp_webp_conversion_status', 'inactive' );

		wp_send_json_success( array(
			'progress' => $progress,
			'status'   => $status,
		) );
	}

	/**
	 * Test single image conversion.
	 *
	 * @since 1.0.0
	 */
	public function test_conversion() {
		check_ajax_referer( 'webp_bulk_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'Mak8it.com' ) ) );
		}

		// This would handle file upload and testing
		// Implementation similar to test-conversion.php page
		
		wp_send_json_success( array( 'message' => __( 'Test completed', 'Mak8it.com' ) ) );
	}

	/**
	 * Check server compatibility.
	 *
	 * @since 1.0.0
	 */
	public function check_server() {
		check_ajax_referer( 'webp_bulk_nonce', 'nonce' );

		$support = WP_WebP_Optimizer_Converter::check_webp_support();

		$info = array(
			'webp_support'  => $support,
			'php_version'   => PHP_VERSION,
			'memory_limit'  => ini_get( 'memory_limit' ),
			'max_upload'    => size_format( wp_max_upload_size() ),
			'gd_loaded'     => extension_loaded( 'gd' ),
			'imagick_loaded' => extension_loaded( 'imagick' ),
		);

		wp_send_json_success( $info );
	}

	/**
	 * Restore original images.
	 *
	 * @since 1.0.0
	 */
	public function restore_originals() {
		check_ajax_referer( 'webp_bulk_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'Mak8it.com' ) ) );
		}

		// Get all images with backups
		global $wpdb;
		$results = $wpdb->get_results(
			"SELECT post_id, meta_value FROM {$wpdb->postmeta} WHERE meta_key = '_webp_backup_path'",
			ARRAY_A
		);

		$restored = 0;
		$failed = 0;

		foreach ( $results as $row ) {
			$image_id = $row['post_id'];
			$backup_path = $row['meta_value'];

			if ( file_exists( $backup_path ) ) {
				$original_path = get_attached_file( $image_id );
				
				if ( copy( $backup_path, $original_path ) ) {
					// Delete WebP file
					$webp_path = get_post_meta( $image_id, '_webp_path', true );
					if ( $webp_path && file_exists( $webp_path ) ) {
						wp_delete_file( $webp_path );
					}

					// Update meta
					delete_post_meta( $image_id, '_webp_converted' );
					delete_post_meta( $image_id, '_webp_path' );
					
					$restored++;
				} else {
					$failed++;
				}
			} else {
				$failed++;
			}
		}

		wp_send_json_success( array(
			'message'  => sprintf( __( 'Restored %d images, %d failed', 'Mak8it.com' ), $restored, $failed ),
			'restored' => $restored,
			'failed'   => $failed,
		) );
	}

	/**
	 * Clear conversion logs.
	 *
	 * @since 1.0.0
	 */
	public function clear_logs() {
		check_ajax_referer( 'webp_bulk_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'Mak8it.com' ) ) );
		}

		require_once SMALL_IMAGE_PLUGIN_DIR . 'includes/class-stats.php';
		WP_WebP_Optimizer_Stats::clear_logs();

		wp_send_json_success( array( 'message' => __( 'Logs cleared', 'Mak8it.com' ) ) );
	}

	/**
	 * Export logs to CSV.
	 *
	 * @since 1.0.0
	 */
	public function export_logs() {
		check_ajax_referer( 'webp_bulk_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( __( 'Permission denied', 'Mak8it.com' ) );
		}

		require_once SMALL_IMAGE_PLUGIN_DIR . 'includes/class-stats.php';
		$csv = WP_WebP_Optimizer_Stats::export_logs_csv();

		header( 'Content-Type: text/csv' );
		header( 'Content-Disposition: attachment; filename="webp-conversion-logs-' . date( 'Y-m-d' ) . '.csv"' );
		echo $csv;
		exit;
	}

	/**
	 * Convert single image.
	 *
	 * @since 1.0.0
	 */
	public function convert_single() {
		check_ajax_referer( 'webp_bulk_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'Mak8it.com' ) ) );
		}

		$image_id = isset( $_POST['image_id'] ) ? intval( $_POST['image_id'] ) : 0;

		if ( ! $image_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid image ID', 'Mak8it.com' ) ) );
		}

		$file_path = get_attached_file( $image_id );

		if ( ! $file_path || ! file_exists( $file_path ) ) {
			wp_send_json_error( array( 'message' => __( 'File not found', 'Mak8it.com' ) ) );
		}

		require_once SMALL_IMAGE_PLUGIN_DIR . 'includes/class-converter.php';
		$converter = new WP_WebP_Optimizer_Converter();
		
		$settings = get_option( 'wp_webp_optimizer_settings', array() );
		$quality = isset( $settings['quality'] ) ? intval( $settings['quality'] ) : 60;

		$result = $converter->convert_image_to_webp( $file_path, $quality );

		if ( $result['success'] ) {
			// Update meta
			$original_size = filesize( $file_path );
			$webp_size = file_exists( $result['webp_path'] ) ? filesize( $result['webp_path'] ) : 0;

			update_post_meta( $image_id, '_webp_converted', '1' );
			update_post_meta( $image_id, '_webp_original_size', $original_size );
			update_post_meta( $image_id, '_webp_new_size', $webp_size );
			update_post_meta( $image_id, '_webp_path', $result['webp_path'] );

			wp_send_json_success( array(
				'message'       => $result['message'],
				'original_size' => size_format( $original_size, 2 ),
				'webp_size'     => size_format( $webp_size, 2 ),
				'savings'       => size_format( $original_size - $webp_size, 2 ),
			) );
		} else {
			wp_send_json_error( array( 'message' => $result['message'] ) );
		}
	}
}


